<?php

namespace App\Http\Controllers;

use App\Models\AvailableFor;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class AvailableForController extends Controller
{
    public function index(): JsonResponse
    {
        $availabilities = AvailableFor::with(['candidateProfile'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $availabilities,
        ], 200);
    }

    public function show($profileId): JsonResponse
    {
        $availability = AvailableFor::where('profile_id', $profileId)->first();

        if (!$availability) {
            return response()->json([
                'status' => 'error',
                'message' => 'Availability not found for this profile',
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => $availability,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        // Disable store method, use PUT instead
        return response()->json([
            'status' => 'error',
            'message' => 'Use PUT request to /api/available-for/{profileId} for create/update operations.',
        ], 405); // Method Not Allowed
    }

    public function update(Request $request, $profileId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'service_type' => 'sometimes|array',
            'service_type.*' => 'in:regular_babysitting,occasional_babysitting,after_school_care,full_time_nanny',
        ], [
            'service_type.*.in' => 'Each service type must be regular_babysitting, occasional_babysitting, after_school_care, or full_time_nanny.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $serviceType = $request->service_type ?? [];

        // Fetch availability for the profile
        $availability = AvailableFor::where('profile_id', $profileId)->first();

        if ($availability) {
            if ($request->has('service_type') && !empty($serviceType)) {
                // If new service_type is provided, delete existing and create new
                $availability->delete();
                $responseData = AvailableFor::create([
                    'profile_id' => $profileId,
                    'service_type' => array_unique($serviceType),
                ]);
            } else {
                // If no new service_type or empty, merge with existing
                $newServiceTypes = array_unique(array_merge($availability->service_type, $serviceType));
                $availability->update([
                    'service_type' => $newServiceTypes,
                ]);
                $responseData = $availability;
            }
        } else {
            // If no record exists, create a new one
            $responseData = AvailableFor::create([
                'profile_id' => $profileId,
                'service_type' => array_unique($serviceType),
            ]);
        }

        return response()->json([
            'status' => 'success',
            'data' => $responseData,
        ], $availability && !$request->has('service_type') ? 200 : 201); // 201 for create or replace, 200 for merge update
    }

    public function destroy($profileId): JsonResponse
    {
        $availability = AvailableFor::where('profile_id', $profileId)->first();

        if (!$availability) {
            return response()->json([
                'status' => 'error',
                'message' => 'Availability not found for this profile',
            ], 404);
        }

        $availability->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Availability deleted successfully',
        ], 200);
    }
}
